// $Id: CClassManager.cpp,v 1.5 2007/02/08 21:06:44 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CClassManager.hpp"
#include "../IO/CTextStream.hpp"
#include "../Collections/TPointerCollection.hpp"
#include "CClass.hpp"

using Exponent::Basics::CClassManager;
using Exponent::Basics::CClass;
using Exponent::IO::CTextStream;
using Exponent::Collections::TPointerCollection;

//	===========================================================================
CClassManager::CClassManager() : m_classes(0), m_classNuller(0)
{
	// Construct the class storage
	m_classes     = new TPointerCollection<CClass>;
	m_classNuller = new TObjectNuller<CClass>;

	// Now set the default deleter
	m_classes->registerPointerDeleter(m_classNuller);
}

//	===========================================================================
CClassManager::~CClassManager()
{
	FREE_POINTER(m_classes);
	FREE_POINTER(m_classNuller);
}

//	===========================================================================
void CClassManager::registerClass(CClass *theClass)
{
	if (theClass && !this->isClassRegistered(*theClass))
	{
		m_classes->addElement(theClass);
	}
}

//	===========================================================================
bool CClassManager::isClassRegistered(const CClass &theClass) const	
{
	// Check we ar initialised
	if (m_classes == NULL)
	{
		return false;
	}

	// Loop and find
	for (long i = 0; i < m_classes->getArraySize(); i++)
	{
		// Get the class
		CClass *aClass = m_classes->elementAtIndex(i);

		// Check its valid
		if (aClass == NULL)
		{
			continue;
		}

		// Check if its the same
		if (aClass->isTypeOf(theClass))
		{
			return true;
		}
	}

	// Failed to find it
	return false;
}

//	===========================================================================
void CClassManager::logClassInformation(CTextStream *stream) const
{
	// Check we have someting
	if (m_classes == NULL)
	{
		return;
	}

	// Check stream is valid
	if (stream && stream->isStreamOpen() && stream->validForOutput())
	{
		*stream << "+-----------------------------------------------------------------+----------+----------+\n";
		*stream << "| Exponent class information                                      |          |          |\n";
		*stream << "+-----------------------------------------------------------------+----------+----------+\n";
		*stream << "| Class Name                                                      | Instance |   Total  |\n";
		*stream << "+-----------------------------------------------------------------+----------+----------+\n";

		long instance = 0;
		long total    = 0;
		
		// Buffer to print to
		char buffer[1024];

		// Loop through all the elements
		for (long i = 0; i < m_classes->getArraySize(); i++)
		{
			// Get the class
			CClass *aClass = m_classes->elementAtIndex(i);

			// Check its valid
			if (aClass == NULL)
			{
				continue;
			}

#ifndef FULL_CLASS_LOGGING
			// I find that a lot of classes just clogs up the log file, this removes any blank items
			if (aClass->getCurrentInstanceCount() == 0 && aClass->getTotalNumberOfClassConstructions() == 0)
			{
				continue;
			}
#endif

			instance += aClass->getCurrentInstanceCount();
			total    += aClass->getTotalNumberOfClassConstructions();

			// Print the string buffer
			sprintf(buffer, "| %63s |%10li|%10li|\n", aClass->getClassName(), aClass->getCurrentInstanceCount(), aClass->getTotalNumberOfClassConstructions());

			*stream << buffer;
		}

		// Output the footer
		*stream << "+-----------------------------------------------------------------+----------+----------+\n";
		sprintf(buffer, "|                                                                 |%10li|%10li|\n", instance, total);
		*stream << buffer;
		*stream << "+-----------------------------------------------------------------+----------+----------+\n";
	}
}

//	===========================================================================
CClass *CClassManager::getClassWithName(const char *name)
{
	// Loop through all the elements
	for (long i = 0; i < m_classes->getArraySize(); i++)
	{
		// Get the class
		CClass *aClass = m_classes->elementAtIndex(i);

		// Check its valid
		if (aClass == NULL)
		{
			continue;
		}

		// Compare the names
		if (strcmp(aClass->getClassName(), name) == 0)
		{
			return aClass;
		}
	}
	return NULL;
}

//	===========================================================================
void CClassManager::sortClasses()
{
	m_classes->sortArray((TPointerCollection<CClass>::qsortCompare)CClass::compareClasses);
}